/** @file   waypointcontroller.h
 * @brief   Declaration of WaypointController - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */
 
#ifndef H_WWW_WAYPOINTCONTROLLER_H
#define H_WWW_WAYPOINTCONTROLLER_H

#include "basecontroller.h"
#include <vector>
#include "route.h"

namespace WeWantWar {

/** @class  WaypointController
 * @brief   Extends the BaseController with waypoint services.
 * @author  Tomi Lamminsaari
 *
 * This class provides the services that makes it possible to create a
 * controllers that make the GameObject to follow predefined paths.
 */
class WaypointController : public BaseController
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================
  
  /** Possible modes he waypoints are handled. The route can be loopped or
   * it can be followed only once.
   */
  enum Mode {
    /** As the GameObject reaches the final node, it stops there. */
    MODE_ONCE,
    /** As the GameObject reaches the final node it heads towards the
     * first node and begins a new round.
     */
    MODE_LOOP,
    /** As the GameObject reaches the final node it goes the same route
     * back to first node and then again to the last node.
     */
    MODE_PINGPONG
  };
  


  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   * @param     pObj              Pointer to GameObject we're supposed to
   *                              control.
   */
	WaypointController( GameObject* pObj);


	/** Destructor
   */
	virtual ~WaypointController();


private:
	/** Copy constructor.
   * @param     rO                Reference to another WaypointController
   */
  WaypointController( const WaypointController& rO );

	/** Assignment operator
   * @param     rO                Reference to another WaypointController
   * @return    Reference to us.
   */
  WaypointController& operator = ( const WaypointController& rO );

public:


  ///
  /// Methods
  /// =======
  
  /** Sets the Route this controller should follow.
   * @param     rRoute            Pointer to the route.
   */
  void setRoute( const Route& rRoute );

  /** Sets the mode
   * @param     m                 The mode how the waypoints are handled.
   */
  void setMode( Mode m );
  
  /** Sets the tolerance, how near we have to be the waypoint's center until
   * we decide that we've reached it.
   * @param     tolerance         The tolerance in pixels.
   */
  void setTolerance( float tolerance );
  
  /** Sets the index of the waypoint where we should be heading next.
   * @param     nextWP            Index of the next waypoint
   */
  void setNextWaypoint( int nextWP );
  
  /** Sets the direction which way we go the waypoints through.
   * @param     dir               Direction. 1 = from first to last
   *                              -1 = from last to first
   */
  void direction( int dir );
  

  /** Changes the active waypoint to next waypoint. This manages the loopping
   * and pingponging.
   */
  void step();
  

  ///
  /// Getter methods
  /// ==============
  
  /** Returns the index of the waypoint we're currently heading towards.
   * @return    Index of next waypoint
   */
  int getActiveWaypoint() const;
  
  /** Returns the waypoint mode.
   * @return    Current waypoint mode
   */
  Mode getMode() const;
  
  /** Returns the maximum distance until we accept that we've reached the
   * waypointnode
   * @return    Tolerance in pixels
   */
  float getTolerance() const;
  
  /** Returns the current direction we're going in our route.
   * @return    1 if we're going from first waypoint to the last. -1 if
   *            opposite direction.
   */
  int direction() const;
  
  /** Returns the route.
   * @return    Pointer to Route
   */
  const Route&  getRoute() const;
  
  /** Tells if we're having a valid route. Valid route has more that 1
   * waypoints.
   * @return    <code>true</code> if our route has more than 1 waypoints.
   */
  bool validRoute() const;
  

protected:

  ///
  /// Protected methods
  /// =================
  
  /** Tells if we've reached the current waypoint.
   * @return    <code>true</code> if we've reached it and we should start
   *            heading towards the next waypoint.
   */
  bool waypointReached() const;
  
  /** Tells if the nose of our GameObject points towards the current waypoint.
   * @param     factor            The factor that defines how exactly we must
   *                              be pointing towards it.
   * @return    <code>true</code> if nose points towards the waypoint.
   */
  bool isFacingWaypoint( float factor) const;
  
  /** Find the shortest turning direction if we're wanting to turn towards the
   * current waypoint.
   * @return    -1 = counter-clockwise, 1 = clockwise, 0 = straight ahead
   */
  int findShortestTurn() const;
  
  

  ///
  /// Members
  /// =======

  /** The mode the waypoints are handled. */
  Mode  m_waypointmode;

  /** Index of the waypoint we're heading towards. */
  int   m_activeWaypoint;
  
  /** Tolerance, how near we have to be the WaypointNode until we have
   * reached it.
   */
  float m_tolerance;
  
  /** The advance direction. */
  int m_dir;
  
  /** The route this controller should follow */
  Route m_route;
  
private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: waypointcontroller.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:20+02  lamminsa
 * Initial revision
 *
 */
 
